// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_fuse.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_fuse__A0__F385

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp32@16_48_0_F_F_1___gfx950)xyzw" };
#else
{ R"xyzw(amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp32@16_48_0_F_F_1___gfx950)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊fp32@16_48_0_F_F_1)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx950)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0x02bbbfddu, 0xa765246au, 0, 6 }, // 02bbbfdda765246a = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp32@16_48_0_F_F_1___gfx950__P__16_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0x1e0f0427u, 0x67c36c2au, 0, 143 }, // 1e0f042767c36c2a = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp32@16_48_0_F_F_1___gfx950__P__16_16__CO__wave3_warp2_stg1--Arch_gfx950
    { 0x14ad49e1u, 0x5e081f20u, 0, 29 }, // 14ad49e15e081f20 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp32@16_48_0_F_F_1___gfx950__P__16_16__CO__wave3_warp4_stg1--Arch_gfx950
    { 0xfa0b424bu, 0xbc80fcfeu, 0, 86 }, // fa0b424bbc80fcfe = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp32@16_48_0_F_F_1___gfx950__P__16_16__CO__wave4_warp2_stg1--Arch_gfx950
    { 0xc52f6d6bu, 0x570da62bu, 23, 109 }, // c52f6d6b570da62b = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp32@16_48_0_F_F_1___gfx950__P__32_16__CO__wave1_warp2_stg1--Arch_gfx950
    { 0x4e5cfd66u, 0xe6c94eebu, 23, 126 }, // 4e5cfd66e6c94eeb = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp32@16_48_0_F_F_1___gfx950__P__32_16__CO__wave1_warp4_stg1--Arch_gfx950
    { 0xc481947au, 0x86339be6u, 23, 6 }, // c481947a86339be6 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp32@16_48_0_F_F_1___gfx950__P__32_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0xe2dab312u, 0x18455751u, 23, 69 }, // e2dab31218455751 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp32@16_48_0_F_F_1___gfx950__P__32_16__CO__wave2_warp4_stg1--Arch_gfx950
    { 0xaa5a26e3u, 0x3ddd7641u, 23, 143 }, // aa5a26e33ddd7641 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp32@16_48_0_F_F_1___gfx950__P__32_16__CO__wave3_warp2_stg1--Arch_gfx950
    { 0x82deea25u, 0x45cc4c1au, 23, 29 }, // 82deea2545cc4c1a = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp32@16_48_0_F_F_1___gfx950__P__32_16__CO__wave3_warp4_stg1--Arch_gfx950
    { 0x3b60a3a7u, 0x3ccc5fa6u, 63, 109 }, // 3b60a3a73ccc5fa6 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp32@16_48_0_F_F_1___gfx950__P__32_32__CO__wave1_warp2_stg1--Arch_gfx950
    { 0xc103dad3u, 0x46a767c5u, 63, 6 }, // c103dad346a767c5 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp32@16_48_0_F_F_1___gfx950__P__32_32__CO__wave2_warp2_stg1--Arch_gfx950
    { 0x7f84fa69u, 0x6c2e47e6u, 63, 69 }, // 7f84fa696c2e47e6 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp32@16_48_0_F_F_1___gfx950__P__32_32__CO__wave2_warp4_stg1--Arch_gfx950
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_fuse_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_fuse_packed_string);

static int8_t lut[1][7][7] =
{
// GPU gfx950_mod0
{{ 7, 7, 7, 6, 3, 6, 6},
 { 7, 2, 6, 6, 3, 6, 6},
 { 9, 6, 5, 0,11, 6, 6},
 { 6, 6, 1, 0, 1, 6, 6},
 { 6, 6,11, 6, 0,11,11},
 { 6, 6, 6, 6,11,11,11},
 { 6, 6, 6, 6,11,11,11}}
// End of GPU gfx950_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelFuseContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelFuseContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_fuse__A0__F385 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 1;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_fuse__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 1;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*fp32:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = 0
// philox_seed_ptr = 0
// philox_offset1 = 0
// philox_offset2 = 0
// Window_left = 0
// BLOCK_DMODEL = 48
// CAUSAL_TYPE = 0
// ENABLE_DROPOUT = False
// PADDED_HEAD = False
// BIAS_TYPE = 1

// SELECT * FROM FLASH$bwd_kernel_fuse WHERE gpu IN ('gfx950_mod0') AND inputs$Q_dtype = 'torch.float32' AND inputs$BLOCK_DMODEL = 48 AND inputs$CAUSAL_TYPE = 0 AND inputs$ENABLE_DROPOUT = False AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 1

// vim: set fileencoding=utf-8

