{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- @GIOStream@ represents an object that has both read and write streams.
-- Generally the two streams act as separate input and output streams,
-- but they share some common resources and state. For instance, for
-- seekable streams, both streams may use the same position.
-- 
-- Examples of @GIOStream@ objects are t'GI.Gio.Objects.SocketConnection.SocketConnection', which represents
-- a two-way network connection; and t'GI.Gio.Objects.FileIOStream.FileIOStream', which represents a
-- file handle opened in read-write mode.
-- 
-- To do the actual reading and writing you need to get the substreams
-- with 'GI.Gio.Objects.IOStream.iOStreamGetInputStream' and
-- 'GI.Gio.Objects.IOStream.iOStreamGetOutputStream'.
-- 
-- The @GIOStream@ object owns the input and the output streams, not the other
-- way around, so keeping the substreams alive will not keep the @GIOStream@
-- object alive. If the @GIOStream@ object is freed it will be closed, thus
-- closing the substreams, so even if the substreams stay alive they will
-- always return @G_IO_ERROR_CLOSED@ for all operations.
-- 
-- To close a stream use 'GI.Gio.Objects.IOStream.iOStreamClose' which will close the common
-- stream object and also the individual substreams. You can also close
-- the substreams themselves. In most cases this only marks the
-- substream as closed, so further I\/O on it fails but common state in the
-- @GIOStream@ may still be open. However, some streams may support
-- ‘half-closed’ states where one direction of the stream is actually shut down.
-- 
-- Operations on @GIOStream@s cannot be started while another operation on the
-- @GIOStream@ or its substreams is in progress. Specifically, an application can
-- read from the t'GI.Gio.Objects.InputStream.InputStream' and write to the
-- t'GI.Gio.Objects.OutputStream.OutputStream' simultaneously (either in separate threads, or as
-- asynchronous operations in the same thread), but an application cannot start
-- any @GIOStream@ operation while there is a @GIOStream@, @GInputStream@ or
-- @GOutputStream@ operation in progress, and an application can’t start any
-- @GInputStream@ or @GOutputStream@ operation while there is a @GIOStream@
-- operation in progress.
-- 
-- This is a product of individual stream operations being associated with a
-- given [type/@gLib@/.MainContext] (the thread-default context at the time the
-- operation was started), rather than entire streams being associated with a
-- single @GMainContext@.
-- 
-- GIO may run operations on @GIOStream@s from other (worker) threads, and this
-- may be exposed to application code in the behaviour of wrapper streams, such
-- as t'GI.Gio.Objects.BufferedInputStream.BufferedInputStream' or t'GI.Gio.Objects.TlsConnection.TlsConnection'. With such
-- wrapper APIs, application code may only run operations on the base (wrapped)
-- stream when the wrapper stream is idle. Note that the semantics of such
-- operations may not be well-defined due to the state the wrapper stream leaves
-- the base stream in (though they are guaranteed not to crash).
-- 
-- /Since: 2.22/

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gio.Objects.IOStream
    ( 

-- * Exported types
    IOStream(..)                            ,
    IsIOStream                              ,
    toIOStream                              ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [clearPending]("GI.Gio.Objects.IOStream#g:method:clearPending"), [close]("GI.Gio.Objects.IOStream#g:method:close"), [closeAsync]("GI.Gio.Objects.IOStream#g:method:closeAsync"), [closeFinish]("GI.Gio.Objects.IOStream#g:method:closeFinish"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [hasPending]("GI.Gio.Objects.IOStream#g:method:hasPending"), [isClosed]("GI.Gio.Objects.IOStream#g:method:isClosed"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [spliceAsync]("GI.Gio.Objects.IOStream#g:method:spliceAsync"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getData]("GI.GObject.Objects.Object#g:method:getData"), [getInputStream]("GI.Gio.Objects.IOStream#g:method:getInputStream"), [getOutputStream]("GI.Gio.Objects.IOStream#g:method:getOutputStream"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata").
-- 
-- ==== Setters
-- [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setPending]("GI.Gio.Objects.IOStream#g:method:setPending"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty").

#if defined(ENABLE_OVERLOADING)
    ResolveIOStreamMethod                   ,
#endif

-- ** clearPending #method:clearPending#

#if defined(ENABLE_OVERLOADING)
    IOStreamClearPendingMethodInfo          ,
#endif
    iOStreamClearPending                    ,


-- ** close #method:close#

#if defined(ENABLE_OVERLOADING)
    IOStreamCloseMethodInfo                 ,
#endif
    iOStreamClose                           ,


-- ** closeAsync #method:closeAsync#

#if defined(ENABLE_OVERLOADING)
    IOStreamCloseAsyncMethodInfo            ,
#endif
    iOStreamCloseAsync                      ,


-- ** closeFinish #method:closeFinish#

#if defined(ENABLE_OVERLOADING)
    IOStreamCloseFinishMethodInfo           ,
#endif
    iOStreamCloseFinish                     ,


-- ** getInputStream #method:getInputStream#

#if defined(ENABLE_OVERLOADING)
    IOStreamGetInputStreamMethodInfo        ,
#endif
    iOStreamGetInputStream                  ,


-- ** getOutputStream #method:getOutputStream#

#if defined(ENABLE_OVERLOADING)
    IOStreamGetOutputStreamMethodInfo       ,
#endif
    iOStreamGetOutputStream                 ,


-- ** hasPending #method:hasPending#

#if defined(ENABLE_OVERLOADING)
    IOStreamHasPendingMethodInfo            ,
#endif
    iOStreamHasPending                      ,


-- ** isClosed #method:isClosed#

#if defined(ENABLE_OVERLOADING)
    IOStreamIsClosedMethodInfo              ,
#endif
    iOStreamIsClosed                        ,


-- ** setPending #method:setPending#

#if defined(ENABLE_OVERLOADING)
    IOStreamSetPendingMethodInfo            ,
#endif
    iOStreamSetPending                      ,


-- ** spliceAsync #method:spliceAsync#

#if defined(ENABLE_OVERLOADING)
    IOStreamSpliceAsyncMethodInfo           ,
#endif
    iOStreamSpliceAsync                     ,


-- ** spliceFinish #method:spliceFinish#

    iOStreamSpliceFinish                    ,




 -- * Properties


-- ** closed #attr:closed#
-- | Whether the stream is closed.
-- 
-- /Since: 2.22/

#if defined(ENABLE_OVERLOADING)
    IOStreamClosedPropertyInfo              ,
#endif
    getIOStreamClosed                       ,
#if defined(ENABLE_OVERLOADING)
    iOStreamClosed                          ,
#endif


-- ** inputStream #attr:inputStream#
-- | The t'GI.Gio.Objects.InputStream.InputStream' to read from.
-- 
-- /Since: 2.22/

#if defined(ENABLE_OVERLOADING)
    IOStreamInputStreamPropertyInfo         ,
#endif
    getIOStreamInputStream                  ,
#if defined(ENABLE_OVERLOADING)
    iOStreamInputStream                     ,
#endif


-- ** outputStream #attr:outputStream#
-- | The t'GI.Gio.Objects.OutputStream.OutputStream' to write to.
-- 
-- /Since: 2.22/

#if defined(ENABLE_OVERLOADING)
    IOStreamOutputStreamPropertyInfo        ,
#endif
    getIOStreamOutputStream                 ,
#if defined(ENABLE_OVERLOADING)
    iOStreamOutputStream                    ,
#endif




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.GLib.Callbacks as GLib.Callbacks
import qualified GI.GLib.Structs.Bytes as GLib.Bytes
import qualified GI.GLib.Structs.PollFD as GLib.PollFD
import qualified GI.GLib.Structs.Source as GLib.Source
import qualified GI.GObject.Callbacks as GObject.Callbacks
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gio.Callbacks as Gio.Callbacks
import {-# SOURCE #-} qualified GI.Gio.Flags as Gio.Flags
import {-# SOURCE #-} qualified GI.Gio.Interfaces.AsyncResult as Gio.AsyncResult
import {-# SOURCE #-} qualified GI.Gio.Objects.Cancellable as Gio.Cancellable
import {-# SOURCE #-} qualified GI.Gio.Objects.InputStream as Gio.InputStream
import {-# SOURCE #-} qualified GI.Gio.Objects.OutputStream as Gio.OutputStream
import {-# SOURCE #-} qualified GI.Gio.Structs.OutputVector as Gio.OutputVector

#else
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gio.Callbacks as Gio.Callbacks
import {-# SOURCE #-} qualified GI.Gio.Flags as Gio.Flags
import {-# SOURCE #-} qualified GI.Gio.Interfaces.AsyncResult as Gio.AsyncResult
import {-# SOURCE #-} qualified GI.Gio.Objects.Cancellable as Gio.Cancellable
import {-# SOURCE #-} qualified GI.Gio.Objects.InputStream as Gio.InputStream
import {-# SOURCE #-} qualified GI.Gio.Objects.OutputStream as Gio.OutputStream

#endif

-- | Memory-managed wrapper type.
newtype IOStream = IOStream (SP.ManagedPtr IOStream)
    deriving (Eq)

instance SP.ManagedPtrNewtype IOStream where
    toManagedPtr (IOStream p) = p

foreign import ccall "g_io_stream_get_type"
    c_g_io_stream_get_type :: IO B.Types.GType

instance B.Types.TypedObject IOStream where
    glibType = c_g_io_stream_get_type

instance B.Types.GObject IOStream

-- | Type class for types which can be safely cast to t'IOStream', for instance with `toIOStream`.
class (SP.GObject o, O.IsDescendantOf IOStream o) => IsIOStream o
instance (SP.GObject o, O.IsDescendantOf IOStream o) => IsIOStream o

instance O.HasParentTypes IOStream
type instance O.ParentTypes IOStream = '[GObject.Object.Object]

-- | Cast to t'IOStream', for types for which this is known to be safe. For general casts, use 'Data.GI.Base.ManagedPtr.castTo'.
toIOStream :: (MIO.MonadIO m, IsIOStream o) => o -> m IOStream
toIOStream = MIO.liftIO . B.ManagedPtr.unsafeCastTo IOStream

-- | Convert t'IOStream' to and from t'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe IOStream) where
    gvalueGType_ = c_g_io_stream_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr IOStream)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr IOStream)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject IOStream ptr
        else return P.Nothing
        
    

#if defined(ENABLE_OVERLOADING)
type family ResolveIOStreamMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveIOStreamMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolveIOStreamMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolveIOStreamMethod "clearPending" o = IOStreamClearPendingMethodInfo
    ResolveIOStreamMethod "close" o = IOStreamCloseMethodInfo
    ResolveIOStreamMethod "closeAsync" o = IOStreamCloseAsyncMethodInfo
    ResolveIOStreamMethod "closeFinish" o = IOStreamCloseFinishMethodInfo
    ResolveIOStreamMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolveIOStreamMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolveIOStreamMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolveIOStreamMethod "hasPending" o = IOStreamHasPendingMethodInfo
    ResolveIOStreamMethod "isClosed" o = IOStreamIsClosedMethodInfo
    ResolveIOStreamMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolveIOStreamMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolveIOStreamMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolveIOStreamMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolveIOStreamMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolveIOStreamMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolveIOStreamMethod "spliceAsync" o = IOStreamSpliceAsyncMethodInfo
    ResolveIOStreamMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolveIOStreamMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolveIOStreamMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolveIOStreamMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolveIOStreamMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolveIOStreamMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolveIOStreamMethod "getInputStream" o = IOStreamGetInputStreamMethodInfo
    ResolveIOStreamMethod "getOutputStream" o = IOStreamGetOutputStreamMethodInfo
    ResolveIOStreamMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolveIOStreamMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolveIOStreamMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolveIOStreamMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolveIOStreamMethod "setPending" o = IOStreamSetPendingMethodInfo
    ResolveIOStreamMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolveIOStreamMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveIOStreamMethod t IOStream, O.OverloadedMethod info IOStream p) => OL.IsLabel t (IOStream -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveIOStreamMethod t IOStream, O.OverloadedMethod info IOStream p, R.HasField t IOStream p) => R.HasField t IOStream p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveIOStreamMethod t IOStream, O.OverloadedMethodInfo info IOStream) => OL.IsLabel t (O.MethodProxy info IOStream) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

-- VVV Prop "closed"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable]
   -- Nullable: (Just False,Nothing)

-- | Get the value of the “@closed@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' iOStream #closed
-- @
getIOStreamClosed :: (MonadIO m, IsIOStream o) => o -> m Bool
getIOStreamClosed obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "closed"

#if defined(ENABLE_OVERLOADING)
data IOStreamClosedPropertyInfo
instance AttrInfo IOStreamClosedPropertyInfo where
    type AttrAllowedOps IOStreamClosedPropertyInfo = '[ 'AttrGet]
    type AttrBaseTypeConstraint IOStreamClosedPropertyInfo = IsIOStream
    type AttrSetTypeConstraint IOStreamClosedPropertyInfo = (~) ()
    type AttrTransferTypeConstraint IOStreamClosedPropertyInfo = (~) ()
    type AttrTransferType IOStreamClosedPropertyInfo = ()
    type AttrGetType IOStreamClosedPropertyInfo = Bool
    type AttrLabel IOStreamClosedPropertyInfo = "closed"
    type AttrOrigin IOStreamClosedPropertyInfo = IOStream
    attrGet = getIOStreamClosed
    attrSet = undefined
    attrPut = undefined
    attrTransfer _ = undefined
    attrConstruct = undefined
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.IOStream.closed"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-IOStream.html#g:attr:closed"
        })
#endif

-- VVV Prop "input-stream"
   -- Type: TInterface (Name {namespace = "Gio", name = "InputStream"})
   -- Flags: [PropertyReadable]
   -- Nullable: (Just False,Nothing)

-- | Get the value of the “@input-stream@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' iOStream #inputStream
-- @
getIOStreamInputStream :: (MonadIO m, IsIOStream o) => o -> m Gio.InputStream.InputStream
getIOStreamInputStream obj = MIO.liftIO $ checkUnexpectedNothing "getIOStreamInputStream" $ B.Properties.getObjectPropertyObject obj "input-stream" Gio.InputStream.InputStream

#if defined(ENABLE_OVERLOADING)
data IOStreamInputStreamPropertyInfo
instance AttrInfo IOStreamInputStreamPropertyInfo where
    type AttrAllowedOps IOStreamInputStreamPropertyInfo = '[ 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint IOStreamInputStreamPropertyInfo = IsIOStream
    type AttrSetTypeConstraint IOStreamInputStreamPropertyInfo = (~) ()
    type AttrTransferTypeConstraint IOStreamInputStreamPropertyInfo = (~) ()
    type AttrTransferType IOStreamInputStreamPropertyInfo = ()
    type AttrGetType IOStreamInputStreamPropertyInfo = Gio.InputStream.InputStream
    type AttrLabel IOStreamInputStreamPropertyInfo = "input-stream"
    type AttrOrigin IOStreamInputStreamPropertyInfo = IOStream
    attrGet = getIOStreamInputStream
    attrSet = undefined
    attrPut = undefined
    attrTransfer _ = undefined
    attrConstruct = undefined
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.IOStream.inputStream"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-IOStream.html#g:attr:inputStream"
        })
#endif

-- VVV Prop "output-stream"
   -- Type: TInterface (Name {namespace = "Gio", name = "OutputStream"})
   -- Flags: [PropertyReadable]
   -- Nullable: (Just False,Nothing)

-- | Get the value of the “@output-stream@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' iOStream #outputStream
-- @
getIOStreamOutputStream :: (MonadIO m, IsIOStream o) => o -> m Gio.OutputStream.OutputStream
getIOStreamOutputStream obj = MIO.liftIO $ checkUnexpectedNothing "getIOStreamOutputStream" $ B.Properties.getObjectPropertyObject obj "output-stream" Gio.OutputStream.OutputStream

#if defined(ENABLE_OVERLOADING)
data IOStreamOutputStreamPropertyInfo
instance AttrInfo IOStreamOutputStreamPropertyInfo where
    type AttrAllowedOps IOStreamOutputStreamPropertyInfo = '[ 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint IOStreamOutputStreamPropertyInfo = IsIOStream
    type AttrSetTypeConstraint IOStreamOutputStreamPropertyInfo = (~) ()
    type AttrTransferTypeConstraint IOStreamOutputStreamPropertyInfo = (~) ()
    type AttrTransferType IOStreamOutputStreamPropertyInfo = ()
    type AttrGetType IOStreamOutputStreamPropertyInfo = Gio.OutputStream.OutputStream
    type AttrLabel IOStreamOutputStreamPropertyInfo = "output-stream"
    type AttrOrigin IOStreamOutputStreamPropertyInfo = IOStream
    attrGet = getIOStreamOutputStream
    attrSet = undefined
    attrPut = undefined
    attrTransfer _ = undefined
    attrConstruct = undefined
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.IOStream.outputStream"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-IOStream.html#g:attr:outputStream"
        })
#endif

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList IOStream
type instance O.AttributeList IOStream = IOStreamAttributeList
type IOStreamAttributeList = ('[ '("closed", IOStreamClosedPropertyInfo), '("inputStream", IOStreamInputStreamPropertyInfo), '("outputStream", IOStreamOutputStreamPropertyInfo)] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
iOStreamClosed :: AttrLabelProxy "closed"
iOStreamClosed = AttrLabelProxy

iOStreamInputStream :: AttrLabelProxy "inputStream"
iOStreamInputStream = AttrLabelProxy

iOStreamOutputStream :: AttrLabelProxy "outputStream"
iOStreamOutputStream = AttrLabelProxy

#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList IOStream = IOStreamSignalList
type IOStreamSignalList = ('[ '("notify", GObject.Object.ObjectNotifySignalInfo)] :: [(Symbol, DK.Type)])

#endif

-- method IOStream::clear_pending
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "stream"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "IOStream" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GIOStream" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_io_stream_clear_pending" g_io_stream_clear_pending :: 
    Ptr IOStream ->                         -- stream : TInterface (Name {namespace = "Gio", name = "IOStream"})
    IO ()

-- | Clears the pending flag on /@stream@/.
-- 
-- /Since: 2.22/
iOStreamClearPending ::
    (B.CallStack.HasCallStack, MonadIO m, IsIOStream a) =>
    a
    -- ^ /@stream@/: a t'GI.Gio.Objects.IOStream.IOStream'
    -> m ()
iOStreamClearPending stream = liftIO $ do
    stream' <- unsafeManagedPtrCastPtr stream
    g_io_stream_clear_pending stream'
    touchManagedPtr stream
    return ()

#if defined(ENABLE_OVERLOADING)
data IOStreamClearPendingMethodInfo
instance (signature ~ (m ()), MonadIO m, IsIOStream a) => O.OverloadedMethod IOStreamClearPendingMethodInfo a signature where
    overloadedMethod = iOStreamClearPending

instance O.OverloadedMethodInfo IOStreamClearPendingMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.IOStream.iOStreamClearPending",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-IOStream.html#v:iOStreamClearPending"
        })


#endif

-- method IOStream::close
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "stream"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "IOStream" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GIOStream" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object, %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_io_stream_close" g_io_stream_close :: 
    Ptr IOStream ->                         -- stream : TInterface (Name {namespace = "Gio", name = "IOStream"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Closes the stream, releasing resources related to it. This will also
-- close the individual input and output streams, if they are not already
-- closed.
-- 
-- Once the stream is closed, all other operations will return
-- 'GI.Gio.Enums.IOErrorEnumClosed'. Closing a stream multiple times will not
-- return an error.
-- 
-- Closing a stream will automatically flush any outstanding buffers
-- in the stream.
-- 
-- Streams will be automatically closed when the last reference
-- is dropped, but you might want to call this function to make sure
-- resources are released as early as possible.
-- 
-- Some streams might keep the backing store of the stream (e.g. a file
-- descriptor) open after the stream is closed. See the documentation for
-- the individual stream for details.
-- 
-- On failure the first error that happened will be reported, but the
-- close operation will finish as much as possible. A stream that failed
-- to close will still return 'GI.Gio.Enums.IOErrorEnumClosed' for all operations.
-- Still, it is important to check and report the error to the user,
-- otherwise there might be a loss of data as all data might not be written.
-- 
-- If /@cancellable@/ is not NULL, then the operation can be cancelled by
-- triggering the cancellable object from another thread. If the operation
-- was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned.
-- Cancelling a close will still leave the stream closed, but some streams
-- can use a faster close that doesn\'t block to e.g. check errors.
-- 
-- The default implementation of this method just calls close on the
-- individual input\/output streams.
-- 
-- /Since: 2.22/
iOStreamClose ::
    (B.CallStack.HasCallStack, MonadIO m, IsIOStream a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@stream@/: a t'GI.Gio.Objects.IOStream.IOStream'
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object, 'P.Nothing' to ignore
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
iOStreamClose stream cancellable = liftIO $ do
    stream' <- unsafeManagedPtrCastPtr stream
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        _ <- propagateGError $ g_io_stream_close stream' maybeCancellable
        touchManagedPtr stream
        whenJust cancellable touchManagedPtr
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data IOStreamCloseMethodInfo
instance (signature ~ (Maybe (b) -> m ()), MonadIO m, IsIOStream a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod IOStreamCloseMethodInfo a signature where
    overloadedMethod = iOStreamClose

instance O.OverloadedMethodInfo IOStreamCloseMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.IOStream.iOStreamClose",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-IOStream.html#v:iOStreamClose"
        })


#endif

-- method IOStream::close_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "stream"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "IOStream" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GIOStream" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "io_priority"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the io priority of the request"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "optional cancellable object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a #GAsyncReadyCallback\n  to call when the request is satisfied"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 4
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the data to pass to callback function"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_io_stream_close_async" g_io_stream_close_async :: 
    Ptr IOStream ->                         -- stream : TInterface (Name {namespace = "Gio", name = "IOStream"})
    Int32 ->                                -- io_priority : TBasicType TInt
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Requests an asynchronous close of the stream, releasing resources
-- related to it. When the operation is finished /@callback@/ will be
-- called. You can then call 'GI.Gio.Objects.IOStream.iOStreamCloseFinish' to get
-- the result of the operation.
-- 
-- For behaviour details see 'GI.Gio.Objects.IOStream.iOStreamClose'.
-- 
-- The asynchronous methods have a default fallback that uses threads
-- to implement asynchronicity, so they are optional for inheriting
-- classes. However, if you override one you must override all.
-- 
-- /Since: 2.22/
iOStreamCloseAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsIOStream a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@stream@/: a t'GI.Gio.Objects.IOStream.IOStream'
    -> Int32
    -- ^ /@ioPriority@/: the io priority of the request
    -> Maybe (b)
    -- ^ /@cancellable@/: optional cancellable object
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback'
    --   to call when the request is satisfied
    -> m ()
iOStreamCloseAsync stream ioPriority cancellable callback = liftIO $ do
    stream' <- unsafeManagedPtrCastPtr stream
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_io_stream_close_async stream' ioPriority maybeCancellable maybeCallback userData
    touchManagedPtr stream
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data IOStreamCloseAsyncMethodInfo
instance (signature ~ (Int32 -> Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsIOStream a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod IOStreamCloseAsyncMethodInfo a signature where
    overloadedMethod = iOStreamCloseAsync

instance O.OverloadedMethodInfo IOStreamCloseAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.IOStream.iOStreamCloseAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-IOStream.html#v:iOStreamCloseAsync"
        })


#endif

-- method IOStream::close_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "stream"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "IOStream" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GIOStream" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_io_stream_close_finish" g_io_stream_close_finish :: 
    Ptr IOStream ->                         -- stream : TInterface (Name {namespace = "Gio", name = "IOStream"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Closes a stream.
-- 
-- /Since: 2.22/
iOStreamCloseFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsIOStream a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@stream@/: a t'GI.Gio.Objects.IOStream.IOStream'
    -> b
    -- ^ /@result@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
iOStreamCloseFinish stream result_ = liftIO $ do
    stream' <- unsafeManagedPtrCastPtr stream
    result_' <- unsafeManagedPtrCastPtr result_
    onException (do
        _ <- propagateGError $ g_io_stream_close_finish stream' result_'
        touchManagedPtr stream
        touchManagedPtr result_
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data IOStreamCloseFinishMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsIOStream a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod IOStreamCloseFinishMethodInfo a signature where
    overloadedMethod = iOStreamCloseFinish

instance O.OverloadedMethodInfo IOStreamCloseFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.IOStream.iOStreamCloseFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-IOStream.html#v:iOStreamCloseFinish"
        })


#endif

-- method IOStream::get_input_stream
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "stream"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "IOStream" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GIOStream" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "InputStream" })
-- throws : False
-- Skip return : False

foreign import ccall "g_io_stream_get_input_stream" g_io_stream_get_input_stream :: 
    Ptr IOStream ->                         -- stream : TInterface (Name {namespace = "Gio", name = "IOStream"})
    IO (Ptr Gio.InputStream.InputStream)

-- | Gets the input stream for this object. This is used
-- for reading.
-- 
-- /Since: 2.22/
iOStreamGetInputStream ::
    (B.CallStack.HasCallStack, MonadIO m, IsIOStream a) =>
    a
    -- ^ /@stream@/: a t'GI.Gio.Objects.IOStream.IOStream'
    -> m Gio.InputStream.InputStream
    -- ^ __Returns:__ a t'GI.Gio.Objects.InputStream.InputStream', owned by the t'GI.Gio.Objects.IOStream.IOStream'.
    -- Do not free.
iOStreamGetInputStream stream = liftIO $ do
    stream' <- unsafeManagedPtrCastPtr stream
    result <- g_io_stream_get_input_stream stream'
    checkUnexpectedReturnNULL "iOStreamGetInputStream" result
    result' <- (newObject Gio.InputStream.InputStream) result
    touchManagedPtr stream
    return result'

#if defined(ENABLE_OVERLOADING)
data IOStreamGetInputStreamMethodInfo
instance (signature ~ (m Gio.InputStream.InputStream), MonadIO m, IsIOStream a) => O.OverloadedMethod IOStreamGetInputStreamMethodInfo a signature where
    overloadedMethod = iOStreamGetInputStream

instance O.OverloadedMethodInfo IOStreamGetInputStreamMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.IOStream.iOStreamGetInputStream",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-IOStream.html#v:iOStreamGetInputStream"
        })


#endif

-- method IOStream::get_output_stream
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "stream"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "IOStream" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GIOStream" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gio" , name = "OutputStream" })
-- throws : False
-- Skip return : False

foreign import ccall "g_io_stream_get_output_stream" g_io_stream_get_output_stream :: 
    Ptr IOStream ->                         -- stream : TInterface (Name {namespace = "Gio", name = "IOStream"})
    IO (Ptr Gio.OutputStream.OutputStream)

-- | Gets the output stream for this object. This is used for
-- writing.
-- 
-- /Since: 2.22/
iOStreamGetOutputStream ::
    (B.CallStack.HasCallStack, MonadIO m, IsIOStream a) =>
    a
    -- ^ /@stream@/: a t'GI.Gio.Objects.IOStream.IOStream'
    -> m Gio.OutputStream.OutputStream
    -- ^ __Returns:__ a t'GI.Gio.Objects.OutputStream.OutputStream', owned by the t'GI.Gio.Objects.IOStream.IOStream'.
    -- Do not free.
iOStreamGetOutputStream stream = liftIO $ do
    stream' <- unsafeManagedPtrCastPtr stream
    result <- g_io_stream_get_output_stream stream'
    checkUnexpectedReturnNULL "iOStreamGetOutputStream" result
    result' <- (newObject Gio.OutputStream.OutputStream) result
    touchManagedPtr stream
    return result'

#if defined(ENABLE_OVERLOADING)
data IOStreamGetOutputStreamMethodInfo
instance (signature ~ (m Gio.OutputStream.OutputStream), MonadIO m, IsIOStream a) => O.OverloadedMethod IOStreamGetOutputStreamMethodInfo a signature where
    overloadedMethod = iOStreamGetOutputStream

instance O.OverloadedMethodInfo IOStreamGetOutputStreamMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.IOStream.iOStreamGetOutputStream",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-IOStream.html#v:iOStreamGetOutputStream"
        })


#endif

-- method IOStream::has_pending
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "stream"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "IOStream" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GIOStream" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_io_stream_has_pending" g_io_stream_has_pending :: 
    Ptr IOStream ->                         -- stream : TInterface (Name {namespace = "Gio", name = "IOStream"})
    IO CInt

-- | Checks if a stream has pending actions.
-- 
-- /Since: 2.22/
iOStreamHasPending ::
    (B.CallStack.HasCallStack, MonadIO m, IsIOStream a) =>
    a
    -- ^ /@stream@/: a t'GI.Gio.Objects.IOStream.IOStream'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if /@stream@/ has pending actions.
iOStreamHasPending stream = liftIO $ do
    stream' <- unsafeManagedPtrCastPtr stream
    result <- g_io_stream_has_pending stream'
    let result' = (/= 0) result
    touchManagedPtr stream
    return result'

#if defined(ENABLE_OVERLOADING)
data IOStreamHasPendingMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsIOStream a) => O.OverloadedMethod IOStreamHasPendingMethodInfo a signature where
    overloadedMethod = iOStreamHasPending

instance O.OverloadedMethodInfo IOStreamHasPendingMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.IOStream.iOStreamHasPending",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-IOStream.html#v:iOStreamHasPending"
        })


#endif

-- method IOStream::is_closed
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "stream"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "IOStream" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GIOStream" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_io_stream_is_closed" g_io_stream_is_closed :: 
    Ptr IOStream ->                         -- stream : TInterface (Name {namespace = "Gio", name = "IOStream"})
    IO CInt

-- | Checks if a stream is closed.
-- 
-- /Since: 2.22/
iOStreamIsClosed ::
    (B.CallStack.HasCallStack, MonadIO m, IsIOStream a) =>
    a
    -- ^ /@stream@/: a t'GI.Gio.Objects.IOStream.IOStream'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the stream is closed.
iOStreamIsClosed stream = liftIO $ do
    stream' <- unsafeManagedPtrCastPtr stream
    result <- g_io_stream_is_closed stream'
    let result' = (/= 0) result
    touchManagedPtr stream
    return result'

#if defined(ENABLE_OVERLOADING)
data IOStreamIsClosedMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsIOStream a) => O.OverloadedMethod IOStreamIsClosedMethodInfo a signature where
    overloadedMethod = iOStreamIsClosed

instance O.OverloadedMethodInfo IOStreamIsClosedMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.IOStream.iOStreamIsClosed",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-IOStream.html#v:iOStreamIsClosed"
        })


#endif

-- method IOStream::set_pending
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "stream"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "IOStream" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GIOStream" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_io_stream_set_pending" g_io_stream_set_pending :: 
    Ptr IOStream ->                         -- stream : TInterface (Name {namespace = "Gio", name = "IOStream"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Sets /@stream@/ to have actions pending. If the pending flag is
-- already set or /@stream@/ is closed, it will return 'P.False' and set
-- /@error@/.
-- 
-- /Since: 2.22/
iOStreamSetPending ::
    (B.CallStack.HasCallStack, MonadIO m, IsIOStream a) =>
    a
    -- ^ /@stream@/: a t'GI.Gio.Objects.IOStream.IOStream'
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
iOStreamSetPending stream = liftIO $ do
    stream' <- unsafeManagedPtrCastPtr stream
    onException (do
        _ <- propagateGError $ g_io_stream_set_pending stream'
        touchManagedPtr stream
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data IOStreamSetPendingMethodInfo
instance (signature ~ (m ()), MonadIO m, IsIOStream a) => O.OverloadedMethod IOStreamSetPendingMethodInfo a signature where
    overloadedMethod = iOStreamSetPending

instance O.OverloadedMethodInfo IOStreamSetPendingMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.IOStream.iOStreamSetPending",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-IOStream.html#v:iOStreamSetPending"
        })


#endif

-- method IOStream::splice_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "stream1"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "IOStream" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GIOStream." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "stream2"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "IOStream" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GIOStream." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface
--                 Name { namespace = "Gio" , name = "IOStreamSpliceFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a set of #GIOStreamSpliceFlags."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "io_priority"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the io priority of the request."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object, %NULL to ignore."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a #GAsyncReadyCallback\n  to call when the request is satisfied"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 6
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the data to pass to callback function"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_io_stream_splice_async" g_io_stream_splice_async :: 
    Ptr IOStream ->                         -- stream1 : TInterface (Name {namespace = "Gio", name = "IOStream"})
    Ptr IOStream ->                         -- stream2 : TInterface (Name {namespace = "Gio", name = "IOStream"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "IOStreamSpliceFlags"})
    Int32 ->                                -- io_priority : TBasicType TInt
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Asynchronously splice the output stream of /@stream1@/ to the input stream of
-- /@stream2@/, and splice the output stream of /@stream2@/ to the input stream of
-- /@stream1@/.
-- 
-- When the operation is finished /@callback@/ will be called.
-- You can then call 'GI.Gio.Objects.IOStream.iOStreamSpliceFinish' to get the
-- result of the operation.
-- 
-- /Since: 2.28/
iOStreamSpliceAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsIOStream a, IsIOStream b, Gio.Cancellable.IsCancellable c) =>
    a
    -- ^ /@stream1@/: a t'GI.Gio.Objects.IOStream.IOStream'.
    -> b
    -- ^ /@stream2@/: a t'GI.Gio.Objects.IOStream.IOStream'.
    -> [Gio.Flags.IOStreamSpliceFlags]
    -- ^ /@flags@/: a set of t'GI.Gio.Flags.IOStreamSpliceFlags'.
    -> Int32
    -- ^ /@ioPriority@/: the io priority of the request.
    -> Maybe (c)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object, 'P.Nothing' to ignore.
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback'
    --   to call when the request is satisfied
    -> m ()
iOStreamSpliceAsync stream1 stream2 flags ioPriority cancellable callback = liftIO $ do
    stream1' <- unsafeManagedPtrCastPtr stream1
    stream2' <- unsafeManagedPtrCastPtr stream2
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_io_stream_splice_async stream1' stream2' flags' ioPriority maybeCancellable maybeCallback userData
    touchManagedPtr stream1
    touchManagedPtr stream2
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data IOStreamSpliceAsyncMethodInfo
instance (signature ~ (b -> [Gio.Flags.IOStreamSpliceFlags] -> Int32 -> Maybe (c) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsIOStream a, IsIOStream b, Gio.Cancellable.IsCancellable c) => O.OverloadedMethod IOStreamSpliceAsyncMethodInfo a signature where
    overloadedMethod = iOStreamSpliceAsync

instance O.OverloadedMethodInfo IOStreamSpliceAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.IOStream.iOStreamSpliceAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-IOStream.html#v:iOStreamSpliceAsync"
        })


#endif

-- method IOStream::splice_finish
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_io_stream_splice_finish" g_io_stream_splice_finish :: 
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Finishes an asynchronous io stream splice operation.
-- 
-- /Since: 2.28/
iOStreamSpliceFinish ::
    (B.CallStack.HasCallStack, MonadIO m, Gio.AsyncResult.IsAsyncResult a) =>
    a
    -- ^ /@result@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'.
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
iOStreamSpliceFinish result_ = liftIO $ do
    result_' <- unsafeManagedPtrCastPtr result_
    onException (do
        _ <- propagateGError $ g_io_stream_splice_finish result_'
        touchManagedPtr result_
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
#endif


