/* src/fan/mod.rs
 *
 * Copyright 2025 Mission Center Developers
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * SPDX-License-Identifier: GPL-3.0-or-later
 */

use std::path::Path;

use glob::glob;
use magpie_platform::fan::Fan;

use crate::config_dir;

const MK_TO_0_C: u32 = 273150;

pub struct FanCache {
    fans: Vec<Fan>,
    config_dir: Option<Box<Path>>,
}

impl FanCache {
    fn fan_info(&mut self, path: &Path) {
        // read the first glob result for hwmon location
        let parent_dir = path.parent().unwrap();
        let parent_dir_str = path.parent().unwrap().to_str().unwrap();
        let hwmon_idx = if let Some(hwmon_dir) = parent_dir.file_name().unwrap().to_str() {
            hwmon_dir[5..].parse::<u32>().ok().unwrap_or(u32::MAX)
        } else {
            return;
        };

        // read the second glob result for fan index
        let findex = if let Some(hwmon_instance_dir) = path.file_name().unwrap().to_str() {
            hwmon_instance_dir[3..(hwmon_instance_dir.len() - "_input".len())]
                .parse::<u32>()
                .ok()
                .unwrap_or(u32::MAX)
        } else {
            return;
        };

        // do not change the case here. we report how it is, not how it ought to be.
        let fan_label = std::fs::read_to_string(format!("{}/fan{}_label", parent_dir_str, findex))
            .map(|it| it.trim().to_owned())
            .ok()
            .and_then(|it| if it.is_empty() { None } else { Some(it) });
        let temp_label =
            std::fs::read_to_string(format!("{}/temp{}_label", parent_dir_str, findex))
                .map(|it| it.trim().to_owned())
                .ok()
                .and_then(|it| if it.is_empty() { None } else { Some(it) });

        let pwm_percent = match std::fs::read_to_string(format!("{}/pwm{}", parent_dir_str, findex))
        {
            Err(_) => None,
            Ok(it) => it.trim().parse::<u64>().ok().map(|v| v as f32 / 255.),
        };

        let rpm = match std::fs::read_to_string(format!("{}/fan{}_input", parent_dir_str, findex)) {
            Err(_) => return,
            Ok(it) => it.trim().parse::<u32>().ok(),
        };
        let rpm = match rpm {
            None => return,
            Some(rpm) => rpm,
        };

        let max_speed =
            match std::fs::read_to_string(format!("{}/fan{}_max", parent_dir_str, findex)) {
                Err(_) => None,
                Ok(it) => it.trim().parse::<u32>().ok(),
            };

        let temp = match std::fs::read_to_string(format!("{}/temp{}_input", parent_dir_str, findex))
        {
            Err(_) => None,
            Ok(v) => v
                .trim()
                .parse::<i64>()
                .map(|v| (v + MK_TO_0_C as i64) as u32)
                .ok(),
        };

        self.fans.push(Fan {
            fan_index: findex,
            hwmon_index: hwmon_idx,
            fan_label,
            temp_name: temp_label,
            temp_amount: temp,
            rpm,
            pwm_percent: pwm_percent,

            max_rpm: max_speed,
        });
    }
}

impl magpie_platform::fan::FanCache for FanCache {
    fn new() -> Self
    where
        Self: Sized,
    {
        let config_dir = config_dir::build_path("hwmon0");
        Self {
            fans: Vec::new(),
            config_dir,
        }
    }

    fn refresh(&mut self) {
        self.fans.clear();
        let (search_str, error_str) = if let Some(config_dir) = &self.config_dir {
            (
                format!("{}/fan[0-9]*_input", config_dir.display()),
                "config files",
            )
        } else {
            (
                "/sys/class/hwmon/hwmon[0-9]*/fan[0-9]*_input".into(),
                "hwmon entry",
            )
        };
        match glob(&search_str) {
            Ok(globs) => {
                for entry in globs {
                    match entry {
                        Ok(path) => self.fan_info(&path),
                        Err(e) => {
                            log::warn!("Failed to read {}: {}", error_str, e)
                        }
                    }
                }
            }
            Err(e) => {
                log::warn!("Failed to read {}: {}", error_str, e)
            }
        };
    }

    fn cached_entries(&self) -> &[Fan] {
        &self.fans
    }
}

#[cfg(test)]
mod tests {
    use magpie_platform::fan::FanCache;

    #[test]
    fn test_disks_cache() {
        let mut cache = super::FanCache::new();
        cache.refresh();
    }
}
